<?php

namespace App\Services;

use App\Venta;
use App\Cartera;
use App\DetalleCartera;
use App\Articulo;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CarteraService
{
    /**
     * Procesa una venta y crea registros de cartera si es necesario
     */
    public function procesarVenta(Venta $venta, $montoPagado = 0, $metodoPago = 'Efectivo')
    {
        try {
            // Calcular deuda sin modificar la venta
            $montoDeuda = $venta->total - $montoPagado;
            
            // Si hay deuda, crear registro de cartera
            if ($montoDeuda > 0) {
                $this->crearCartera($venta, $montoPagado, $montoDeuda);
            }
            
            return true;
            
        } catch (\Exception $e) {
            \Log::error('Error en CarteraService: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Crea un registro de cartera para una venta con deuda
     */
    private function crearCartera(Venta $venta, $montoPagado, $montoDeuda)
    {
        try {
            // Verificar si ya existe una cartera para esta venta
            $carteraExistente = Cartera::where('idventa_cart', $venta->id)->first();
            
            if ($carteraExistente) {
                return $carteraExistente;
            }
            
            // Crear nueva cartera
            $cartera = new Cartera();
            $cartera->cliente_id = $venta->idcliente;
            $cartera->idventa_cart = $venta->id;
            $cartera->total_deuda = $montoDeuda;
            $cartera->compras = $venta->total;
            $cartera->cuotas = 1;
            $cartera->fecha_compr_pago = $venta->fecha_hora;
            $cartera->total_abono = $montoPagado;
            $cartera->descuento = 0;
            $cartera->devolucion = 0;
            $cartera->saldo_actual = $montoDeuda;
            $cartera->estado_cart = $montoPagado > 0 ? 'Parcial' : 'Pendiente';
            // $cartera->observaciones = "Deuda generada por venta #{$venta->num_comprobante}";
            $cartera->save();
            
            \Log::info('Cartera creada exitosamente: ' . $cartera->id);
            
            // Crear detalles de cartera
            $this->crearDetallesCartera($cartera, $venta, $montoPagado, $montoDeuda);
            
            return $cartera;
        } catch (\Exception $e) {
            \Log::error('Error creando cartera: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Crea los detalles de cartera para cada artículo de la venta
     */
    private function crearDetallesCartera(Cartera $cartera, Venta $venta, $montoPagado, $montoDeuda)
    {
        try {
            // Obtener detalles de venta directamente de la base de datos
            $detalles = \DB::table('detalle_ventas')->where('idventa', $venta->id)->get();
            
            foreach ($detalles as $detalle) {
                $articulo = Articulo::find($detalle->idarticulo);
                
                if (!$articulo) {
                    \Log::warning('Artículo no encontrado: ' . $detalle->idarticulo);
                    continue;
                }
                
                $detalleCartera = new DetalleCartera();
                $detalleCartera->idcartera = $cartera->id;
                $detalleCartera->articulo_id = $detalle->idarticulo;
                $detalleCartera->centrocosto_id = $articulo->centrocosto_id ?? null;
                $detalleCartera->valor_venta_articulo = $articulo->precio_venta;
                $detalleCartera->cantidad = $detalle->cantidad;
                $detalleCartera->compra = $detalle->precio * $detalle->cantidad;
                $detalleCartera->fecha_venta = $venta->fecha_hora;
                $detalleCartera->pago_incompl = $detalle->precio * $detalle->cantidad * ($montoPagado / $venta->total);
                $detalleCartera->deuda = $detalle->precio * $detalle->cantidad * ($montoDeuda / $venta->total);
                $detalleCartera->abono = 0;
                $detalleCartera->descuento = $detalle->descuento;
                $detalleCartera->devolucion = 0;
                $detalleCartera->saldo_actual = $detalleCartera->deuda;
                $detalleCartera->save();
                
                \Log::info('Detalle cartera creado: ' . $detalleCartera->id);
            }
        } catch (\Exception $e) {
            \Log::error('Error creando detalles de cartera: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Registra un abono a la cartera
     */
    public function registrarAbono($carteraId, $montoAbono, $observaciones = null)
    {
        try {
            DB::beginTransaction();
            
            $cartera = Cartera::findOrFail($carteraId);
            
            // Actualizar cartera
            $cartera->total_abono += $montoAbono;
            $cartera->saldo_actual = $cartera->total_deuda - $cartera->total_abono;
            
            // Determinar estado
            if ($cartera->saldo_actual <= 0) {
                $cartera->estado_cart = 'Pagado';
                $cartera->saldo_actual = 0;
            } else {
                $cartera->estado_cart = 'Parcial';
            }
            
            $cartera->save();
            
            // Actualizar detalles de cartera
            $this->actualizarDetallesAbono($cartera, $montoAbono);
            
            DB::commit();
            return $cartera;
            
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
    
    /**
     * Actualiza los detalles de cartera con el abono
     */
    private function actualizarDetallesAbono(Cartera $cartera, $montoAbono)
    {
        $detalles = $cartera->detalles()->where('saldo_actual', '>', 0)->get();
        $totalSaldoPendiente = $detalles->sum('saldo_actual');
        
        foreach ($detalles as $detalle) {
            $proporcion = $detalle->saldo_actual / $totalSaldoPendiente;
            $abonoDetalle = $montoAbono * $proporcion;
            
            $detalle->abono += $abonoDetalle;
            $detalle->fecha_abono = Carbon::now();
            $detalle->saldo_actual = max(0, $detalle->saldo_actual - $abonoDetalle);
            $detalle->save();
        }
    }
    
    /**
     * Obtiene el resumen de cartera de un cliente
     */
    public function obtenerResumenCliente($clienteId)
    {
        $carteras = Cartera::where('cliente_id', $clienteId)
            ->with(['venta', 'detalles.articulo'])
            ->get();
            
        $resumen = [
            'total_deuda' => $carteras->sum('total_deuda'),
            'total_abonado' => $carteras->sum('total_abono'),
            'saldo_actual' => $carteras->sum('saldo_actual'),
            'ventas_pendientes' => $carteras->where('estado_cart', '!=', 'Pagado')->count(),
            'carteras' => $carteras
        ];
        
        return $resumen;
    }
    
    /**
     * Obtiene todas las carteras con filtros
     */
    public function obtenerCarteras($filtros = [])
    {
        $query = Cartera::with(['cliente', 'venta']);
        
        if (isset($filtros['estado'])) {
            $query->where('estado_cart', $filtros['estado']);
        }
        
        if (isset($filtros['cliente_id'])) {
            $query->where('cliente_id', $filtros['cliente_id']);
        }
        
        if (isset($filtros['fecha_desde'])) {
            $query->where('fecha_compr_pago', '>=', $filtros['fecha_desde']);
        }
        
        if (isset($filtros['fecha_hasta'])) {
            $query->where('fecha_compr_pago', '<=', $filtros['fecha_hasta']);
        }
        
        return $query->orderBy('created_at', 'desc')->paginate(10);
    }
} 